/*
 * Copyright (c) 2021, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "ti_msp_dl_config.h"

#define fre_trigger (32.768)                       // trigger clock (LFCLK): 32.768 kHz; source clock: SYSOSC

// 4-bit gray code for RFINE: 0-15
uint32_t RFINE_gray [16] = {0x00,0x01,0x03,0x02,0x06,0x07,0x05,0x04,0x0C,0x0D,0x0F,0x0E,0x0A,0x0B,0x09,0x08};    // Max RFINE value: 0x000F0000
uint32_t RFINE_i_start = 0;
uint32_t RFINE_i_end = 0x0F;
uint32_t RFINE_start = 0x00000000;
uint32_t RFINE_set;

uint32_t RESCOARE_start = 0x00001F00;
uint32_t RESCOARE_end = 0x00003F00;
uint32_t RESCOARE_set;

uint32_t SYSTRIM_start;
uint32_t SYSTRIM_mid;
uint32_t SYSTRIM_end;
uint32_t CAP_set = 0x00000000;
uint32_t freq_usertr = 0x00000002;

uint32_t FCC_period;
uint32_t count_best;
uint32_t count_start;
uint32_t count_end;
uint32_t OSCTRIMU_best;
uint32_t OSCTRIMU_start;
uint32_t OSCTRIMU_end;

float freq_target = 24000;
float freq_mid;
float freq_best;
float freq_start;
float freq_end;
float error_best;
float error_start = 1;
float error_end = 1;

uint32_t perfect_flag;
uint32_t count_validate;

int main(void)
{
    SYSCFG_DL_init();

    // initial trim of SYSOSC frequency and initial counting using FCC
    DL_SYSCTL_setSYSOSCFreq(DL_SYSCTL_SYSOSC_FREQ_BASE);
    DL_Common_updateReg(&SYSCTL->SOCLOCK.SYSOSCTRIMUSER, RESCOARE_start, SYSCTL_SYSOSCTRIMUSER_RESCOARSE_MASK);
    DL_Common_updateReg(&SYSCTL->SOCLOCK.SYSOSCTRIMUSER, RFINE_start, SYSCTL_SYSOSCTRIMUSER_RESFINE_MASK);
    DL_SYSCTL_setSYSOSCFreq(DL_SYSCTL_SYSOSC_FREQ_USERTRIM);
    delay_cycles (10000);

    // Trim SYSOSC using dichotomy method and use FCC to calculate the frequency of SYSOSC
    SYSTRIM_start = (RESCOARE_start >> 4) + RFINE_i_start;
    SYSTRIM_end = (RESCOARE_end >> 4) + RFINE_i_end;

    while (SYSTRIM_end > SYSTRIM_start +1)
    {
       SYSTRIM_mid = (SYSTRIM_end + SYSTRIM_start) / 2;
       RFINE_set = RFINE_gray [SYSTRIM_mid & 0x0000000F] << 16;
       RESCOARE_set = (SYSTRIM_mid & 0x00000FF0) << 4;

       DL_SYSCTL_setSYSOSCFreq(DL_SYSCTL_SYSOSC_FREQ_BASE);
       DL_Common_updateReg(&SYSCTL->SOCLOCK.SYSOSCTRIMUSER, freq_usertr, SYSCTL_SYSOSCTRIMUSER_FREQ_SYSOSC24M);
       DL_Common_updateReg(&SYSCTL->SOCLOCK.SYSOSCTRIMUSER, CAP_set, SYSCTL_SYSOSCTRIMUSER_CAP_MASK);
       DL_Common_updateReg(&SYSCTL->SOCLOCK.SYSOSCTRIMUSER, RESCOARE_set, SYSCTL_SYSOSCTRIMUSER_RESCOARSE_MASK);
       DL_Common_updateReg(&SYSCTL->SOCLOCK.SYSOSCTRIMUSER, RFINE_set, SYSCTL_SYSOSCTRIMUSER_RESFINE_MASK);
       DL_SYSCTL_setSYSOSCFreq(DL_SYSCTL_SYSOSC_FREQ_USERTRIM);
       delay_cycles (10000);

       DL_SYSCTL_startFCC();
       while (false == DL_SYSCTL_isFCCDone());
       OSCTRIMU_best = SYSCTL->SOCLOCK.SYSOSCTRIMUSER;
       FCC_period = (((SYSCTL->SOCLOCK.GENCLKCFG) & 0X1F000000) >> 24) + 1;
       count_best = (SYSCTL->SOCLOCK.FCC) & 0x003FFFFF;
       freq_mid = FCC_period / fre_trigger;
       freq_best = count_best / freq_mid;
       if (freq_best > freq_target) {
           SYSTRIM_start = SYSTRIM_mid;
           freq_start = freq_best;
           error_start = (freq_best - freq_target ) / freq_target * 100;
           count_start =  count_best;
           OSCTRIMU_start = OSCTRIMU_best;
           DL_GPIO_togglePins(GPIO_LEDS_PORT,GPIO_LEDS_USER_T_1_PIN);                              // test point 1: using GPIO PA0
       }
       if (freq_best < freq_target) {
           SYSTRIM_end = SYSTRIM_mid;
           freq_end = freq_best;
           error_end = (freq_target - freq_best) / freq_target * 100;
           count_end = count_best;
           OSCTRIMU_end = OSCTRIMU_best;
           DL_GPIO_togglePins(GPIO_LEDS_PORT,GPIO_LEDS_USER_T_2_PIN);                              // test point 2: using GPIO PA1
       }
       if (freq_best == freq_target) {
           perfect_flag = 1;
           error_best = 0;
           DL_GPIO_setPins(GPIO_LEDS_PORT,GPIO_LEDS_USER_T_3_PIN);                                 // test point 3: using GPIO PA22
           break;
       }
    }


    if (perfect_flag != 1) {
        if (error_end <= error_start) {
            freq_best =  freq_end;
            error_best = error_end;
            count_best = count_end;
            OSCTRIMU_best = OSCTRIMU_end;
        }
        else {
            freq_best =  freq_start;
            error_best = error_start;
            count_best = count_start;
            OSCTRIMU_best = OSCTRIMU_start;
        }
    }
    DL_GPIO_setPins(GPIO_LEDS_PORT,GPIO_LEDS_USER_T_4_PIN);                                        // test point 4: using GPIO PA6

   // using the saved SYSOSCTRIMUSER value to get the 24 MHz SYSOSC
   DL_SYSCTL_setSYSOSCFreq(DL_SYSCTL_SYSOSC_FREQ_BASE);
   SYSCTL->SOCLOCK.SYSOSCTRIMUSER = OSCTRIMU_best;
   DL_SYSCTL_setSYSOSCFreq(DL_SYSCTL_SYSOSC_FREQ_USERTRIM);
   delay_cycles (10000);

   // using FCC to count the source clock (24 MHz SYSOSC) to validate the accuracy
   DL_SYSCTL_startFCC();
   while (false == DL_SYSCTL_isFCCDone());
   count_validate = (SYSCTL->SOCLOCK.FCC) & 0x003FFFFF;
   DL_GPIO_setPins(GPIO_LEDS_PORT,GPIO_LEDS_USER_T_5_PIN);                                         // test point 5: using GPIO PA21

    DL_SYSCTL_disableSleepOnExit();
    while (1) {
        __WFI();
    }
}

